let sorteioRealizado = false;

document.getElementById("btnSortear").addEventListener("click", executarSorteio);

async function executarSorteio() {

    if (sorteioRealizado)
        return erro("Não é possível realizar o sorteio mais de uma vez.");

    const hashInput = document.getElementById("hash").value.trim();
    const fileInput = document.getElementById("arquivo");

    const validacao = validarEntrada(hashInput, fileInput);
    if (!validacao.ok) return erro(validacao.msg);

    try {

        const file = fileInput.files[0];
        const buffer = await file.arrayBuffer();

        const hashArquivo = await gerarHash(buffer);
        if (hashArquivo !== hashInput.toLowerCase())
            return erro("Hash não corresponde ao arquivo.");

        const dados = validarJSON(buffer);
        if (!dados.ok) return erro(dados.msg);

        const dadosSorteio = dados.data;

        const seed = hashArquivo;

        const vencedores = await gerarIndices(
            seed,
            dadosSorteio.bilhetes,
            dadosSorteio.premios.length,
            dadosSorteio.regras.permite_multiplos_premios_por_usuario
        );

        const resultado = montarResultado(dadosSorteio, vencedores);

        const nomeArquivo = gerarNomeArquivo(dadosSorteio.sorteio.descricao);

        const jsonResultado = JSON.stringify(resultado, null, 2);

        const hashResultado = await gerarHash(
            new TextEncoder().encode(jsonResultado)
        );

        mostrarResultadoNaTela(resultado, hashResultado, nomeArquivo);

        sorteioRealizado = true;

        sucesso("Sorteio concluído.");

    } catch (e) {
        erro("Erro ao processar o sorteio.");
    }
}

function validarEntrada(hash, fileInput) {

    if (!hash)
        return { ok: false, msg: "Informe o hash oficial." };

    if (!fileInput.files.length)
        return { ok: false, msg: "Selecione o arquivo." };

    return { ok: true };
}


function validarJSON(buffer) {

    try {
        const texto = new TextDecoder().decode(buffer);
        const data = JSON.parse(texto);

        if (!Array.isArray(data.bilhetes))
            return { ok: false, msg: "Campo 'bilhetes' inválido." };

        if (!Array.isArray(data.premios))
            return { ok: false, msg: "Campo 'premios' inválido." };

        if (typeof data.regras !== "object")
            return { ok: false, msg: "Campo 'regras' inválido." };

        if (typeof data.sorteio !== "object")
            return { ok: false, msg: "Campo 'sorteio' inválido." };

        if (!data.bilhetes.length)
            return { ok: false, msg: "Nenhum bilhete encontrado." };

        return { ok: true, data };

    } catch {
        return { ok: false, msg: "JSON inválido." };
    }
}

async function gerarHash(bufferOuBytes) {

    const hashBuffer = await crypto.subtle.digest("SHA-256", bufferOuBytes);
    const bytes = Array.from(new Uint8Array(hashBuffer));

    return bytes.map(b => b.toString(16).padStart(2, "0")).join("");
}


async function gerarIndices(seed, bilhetes, quantidade, permiteMultiplos) {

    if (quantidade > bilhetes.length)
        throw new Error("Mais prêmios que bilhetes.");

    const vencedores = [];
    let contador = 0;

    let bilhetesDisponiveis = [...bilhetes];

    while (vencedores.length < quantidade) {

        const texto = `${seed}|${contador}`;
        const hash = await gerarHash(new TextEncoder().encode(texto));

        const numero = BigInt("0x" + hash.slice(0, 16));
        const indice = Number(numero % BigInt(bilhetesDisponiveis.length));

        const bilheteSorteado = bilhetesDisponiveis[indice];

        vencedores.push(bilheteSorteado);

        if (!permiteMultiplos) {
            bilhetesDisponiveis = bilhetesDisponiveis.filter(
                b => b.usuario_id !== bilheteSorteado.usuario_id
            );
        } else {
            bilhetesDisponiveis.splice(indice, 1);
        }

        contador++;
    }

    return vencedores;
}

function montarResultado(dados, vencedores) {

    return {
        sorteio_id: dados.sorteio.id,
        descricao: dados.sorteio.descricao,
        comeco_periodo: dados.sorteio.comeco_periodo,
        final_periodo: dados.sorteio.final_periodo,
        data_limite_importacao: dados.sorteio.data_limite_importacao,
        data_do_sorteio: dados.sorteio.data_do_sorteio,

        premios_com_ganhadores: vencedores.map((bilhete, i) => ({
            premio: {
                id: dados.premios[i].id,
                ordem: dados.premios[i].ordem
            },
            bilhete: bilhete.numero
        }))
    };
}


function gerarNomeArquivo(descricao) {

    const nomeLimpo = descricao
        .toLowerCase()
        .replace(/\s+/g, "-");

    return `resultado-sorteio-${nomeLimpo}.json`;
}




function mostrarToast(msg, tipo = "success") {

    const container = document.getElementById("toastContainer");

    const toast = document.createElement("div");
    toast.classList.add("toast");
    toast.classList.add(tipo === "error" ? "toast-error" : "toast-success");

    toast.textContent = msg;

    container.appendChild(toast);

    setTimeout(() => {
        toast.style.opacity = "0";
        toast.style.transform = "translateX(30px)";
        setTimeout(() => toast.remove(), 300);
    }, 4000);
}

function erro(msg) {
    mostrarToast("Erro: " + msg, "error");
}

function sucesso(msg) {
    mostrarToast(msg, "success");
}



function mostrarResultadoNaTela(resultado, hash, nomeArquivo) {

    const area = document.getElementById("resultadoArea");
    const hashEl = document.getElementById("hashResultado");
    const btnDownload = document.getElementById("btnDownloadResultado");
    const btnCopiar = document.getElementById("btnCopiarHash");
    const form = document.getElementById("formSorteio");

    hashEl.textContent = hash;

    area.classList.remove("hidden");

    form.style.display = "none";

    btnDownload.onclick = () => {
        baixarJSON(resultado, nomeArquivo);
    };

    btnCopiar.onclick = async () => {
        await navigator.clipboard.writeText(hash);
        btnCopiar.textContent = "Copiado ✓";

        setTimeout(() => {
            btnCopiar.textContent = "Copiar";
        }, 2000);
    };
}

function baixarJSON(obj, nome) {

    const json = JSON.stringify(obj, null, 2);

    const blob = new Blob([json], { type: "application/json" });
    const url = URL.createObjectURL(blob);

    const a = document.createElement("a");
    a.href = url;
    a.download = nome;
    a.click();

    URL.revokeObjectURL(url);

    return json;
}